﻿
/****************************************************************************/
/*Copyright (c) 2011, Florent DEVILLE.                                      */
/*All rights reserved.                                                      */
/*                                                                          */
/*Redistribution and use in source and binary forms, with or without        */
/*modification, are permitted provided that the following conditions        */
/*are met:                                                                  */
/*                                                                          */
/* - Redistributions of source code must retain the above copyright         */
/*notice, this list of conditions and the following disclaimer.             */
/* - Redistributions in binary form must reproduce the above                */
/*copyright notice, this list of conditions and the following               */
/*disclaimer in the documentation and/or other materials provided           */
/*with the distribution.                                                    */
/* - The names of its contributors cannot be used to endorse or promote     */
/*products derived from this software without specific prior written        */
/*permission.                                                               */
/* - The source code cannot be used for commercial purposes without         */
/*its contributors' permission.                                             */
/*                                                                          */
/*THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS       */
/*"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT         */
/*LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS         */
/*FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE            */
/*COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,       */
/*INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,      */
/*BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;          */
/*LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER          */
/*CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT        */
/*LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN         */
/*ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE           */
/*POSSIBILITY OF SUCH DAMAGE.                                               */
/****************************************************************************/

using GE.Visualisation;
using GE.Physics.Shapes;
using GE.Physics;
using Microsoft.Xna.Framework;

namespace GE.World.Entities
{
    /// <summary>
    /// A napalm enemy launching shrapnel when it explodes
    /// </summary>
    class NapalmEntity : EnemyEntity
    {
        enum eState
        {
            eStateNapalm,
            eStateWaitRespawn,
            eStateCount
        }

        /// <summary>
        /// Inner state
        /// </summary>
        eState _innerState;

        /// <summary>
        /// Time to wait before the entity respawn
        /// </summary>
        int _iRespawnTime;

        /// <summary>
        /// Last time the entity respawned
        /// </summary>
        int _iRespawnLastTime;

        /// <summary>
        /// Texture's id
        /// </summary>
        int _iIdTexture;

        /// <summary>
        /// Sprite's id
        /// </summary>
        int _iIdSprite;

        /// <summary>
        /// Napalm Shape
        /// </summary>
        //DynamicShapeRectangle _shape;

        /// <summary>
        /// Napalm's maximum vertical speed
        /// </summary>
        int _iMaxSpeed;

        /// <summary>
        /// Napalm life time
        /// </summary>
        int _iLifeTime;

        /// <summary>
        /// Napalm's start position
        /// </summary>
        Vector2 _v2StartPosition;

        /// <summary>
        /// Napalm's birth time
        /// </summary>
        int _iNapalmBirthTime;

        /// <summary>
        /// Flag set when the inner state changes
        /// </summary>
        bool _bPreState;

        const int SHRAPNEL_COUNT = 4;

        /// <summary>
        /// Arrays of shrapnel entities
        /// </summary>
        ShrapnelEntity[] _shrapnel;

        /// <summary>
        /// Id of the explosion animation
        /// </summary>
        int _iIdAnimationExplosion;

        /// <summary>
        /// Position offset of the animation
        /// </summary>
        Vector2 _v2AnimationOffset;

        /// <summary>
        /// Tha damages for the shrapnel
        /// </summary>
        int _iShrapnelDamages;

        #region Properties
#if !GAME
        public static string EDITOR_TILESET { get { return "enemysheet.xml"; } }
        public static string EDITOR_SPRITE { get { return "napalm"; } }
#endif
        public int RespawnTime { set { _iRespawnTime = value; } }
        public int LifeTime { set { _iLifeTime = value; } }
        public int MaxSpeed { set { _iMaxSpeed = value; } }
        public int ShrapnelDamages { set { _iShrapnelDamages = value; } }

        #endregion

        /// <summary>
        /// Constructor
        /// </summary>
        public NapalmEntity()
            : base()
        {
            _iIdTexture = -1;
            _iIdSprite = -1;

            _shape = Physics.Physics.Instance.createDynamicRectangle(0, 0, Vector2.Zero, this);
            _shape._bCollisionEnable = false;
            _shape._iGroup = (int)ePhysicGroup.ePhysicEnemy;
            _shrapnel = new ShrapnelEntity[SHRAPNEL_COUNT];
            for (int i = 0; i < SHRAPNEL_COUNT; i++)
                _shrapnel[i] = World.Instance.createBlankShrapnel();

            _iRespawnTime = -1;
            _iRespawnLastTime = -1;
            _iShrapnelDamages = -1;
        }

        /// <summary>
        /// Initiator : Initialise the entity. Load all the data which are not loaded during the level loading.
        /// </summary>
        public override void init()
        {
            _iIdTexture = Visu.Instance.loadTilset("enemysheet.xml");
            _iIdSprite = Visu.Instance.getSpriteId(_iIdTexture, "napalm");
            _iIdAnimationExplosion = Visu.Instance.getAnimationID("Little_Explosion");

            _shape.resize(Visu.Instance.getSpriteWidth(_iIdTexture, _iIdSprite), Visu.Instance.getSpriteHeight(_iIdTexture, _iIdSprite));
            _shape._iGroup = (int)ePhysicGroup.ePhysicEnemy;

            int width = Visu.Instance.getSpriteWidth(_iIdTexture, _iIdSprite);
            int height = Visu.Instance.getSpriteHeight(_iIdTexture, _iIdSprite);
            _v2AnimationOffset = new Vector2(width / 2, height / 2);

            base.init();
        }

        /// <summary>
        /// Activator
        /// </summary>
        /// <param name="position"></param>
        public override void activate()
        {
            //set the shapes
            _shape._bCollisionEnable = true;
            _shape._v2position = _v2Position;
            _innerState = eState.eStateNapalm;
            _v2StartPosition = _v2Position;
            
            _bPreState = true;
            _bActive = true;

        }

        public override void update()
        {
            switch (_innerState)
            {
                case eState.eStateNapalm:
                    updateStateNapalm();
                    break;

                case eState.eStateWaitRespawn:
                    updateStateWaitRespawn();
                    break;
                        
            }

            CollisionResult res = Physics.Physics.Instance.checkFirstRegisteredCollisionEx(_shape, (int)ePhysicGroup.ePhysicPlayer);
            if (res != null)
            {
                World.Instance.PlayerEntity.hurt(_iDamages);
            }
        }

        public void updateStateNapalm()
        {
            if (_bPreState)
            {
                _bPreState = false;
                _iNapalmBirthTime = TimeClock.Clock.instance.millisecs;
                _v2Position = _v2StartPosition;
            }

            float fElapsedTime = TimeClock.Clock.instance.millisecs - _iNapalmBirthTime;
            float fRatio = fElapsedTime / _iLifeTime;
            float fSpeed = (1f - fRatio) * _iMaxSpeed;
            _v2Position.Y -= fSpeed;
            _shape._v2position = _v2Position;

            if (fElapsedTime >= _iLifeTime)
            {
                const int Y = 5;
                _shrapnel[0].activate(_v2Position, new Vector2(2, Y));
                _shrapnel[0].Damages = _iShrapnelDamages;
                _shrapnel[1].activate(_v2Position, new Vector2(-2, Y));
                _shrapnel[1].Damages = _iShrapnelDamages;
                _shrapnel[2].activate(_v2Position, new Vector2(3, Y));
                _shrapnel[2].Damages = _iShrapnelDamages;
                _shrapnel[3].activate(_v2Position, new Vector2(-3, Y));
                _shrapnel[3].Damages = _iShrapnelDamages;

                die();
            }

        }

        public void updateStateWaitRespawn()
        {
            if (_bPreState)
            {
                _iRespawnLastTime = TimeClock.Clock.instance.millisecs;
                _bPreState = false;
                _v2Position = _v2StartPosition;
                _shape._v2position = _v2StartPosition;
            }

            if (TimeClock.Clock.instance.millisecs >= _iRespawnLastTime + _iRespawnTime)
            {
                _innerState = eState.eStateNapalm;
                _bPreState = true;
            }
        }

        public override void render()
        {
            switch (_innerState)
            {
                case eState.eStateNapalm:
                    Visu.Instance.displaySprite(_iIdTexture, _iIdSprite, ScreenPosition);
                    break;
                
            }
        }

        public override void die()
        {
            _innerState = eState.eStateWaitRespawn;
            _bPreState = true;

            Manager.ExplosionManager.Instance.activate(_iIdAnimationExplosion, Position + _v2AnimationOffset);
        }

        //public override void hurt(int damages)
        //{
        //    die();
        //}
    }
}
